/*
 * Decompiled with CFR 0.152.
 */
package com.admc.jcreole;

import beaver.Action;
import beaver.Parser;
import beaver.ParsingTables;
import beaver.Symbol;
import com.admc.jcreole.CreoleParseException;
import com.admc.jcreole.DtSymbol;
import com.admc.jcreole.EntryOrdering;
import com.admc.jcreole.EntrySymbol;
import com.admc.jcreole.EntryType;
import com.admc.jcreole.InterWikiMapper;
import com.admc.jcreole.JCreolePrivilege;
import com.admc.jcreole.LiSymbol;
import com.admc.jcreole.Sections;
import com.admc.jcreole.TabSymbol;
import com.admc.jcreole.TagType;
import com.admc.jcreole.Token;
import com.admc.jcreole.WashedSymbol;
import com.admc.jcreole.marker.BlockMarker;
import com.admc.jcreole.marker.CloseMarker;
import com.admc.jcreole.marker.DeferredUrlMarker;
import com.admc.jcreole.marker.FootNoteRefMarker;
import com.admc.jcreole.marker.FootNotesMarker;
import com.admc.jcreole.marker.HeadingMarker;
import com.admc.jcreole.marker.IndexMarker;
import com.admc.jcreole.marker.IndexedMarker;
import com.admc.jcreole.marker.InlineMarker;
import com.admc.jcreole.marker.JcxBlockMarker;
import com.admc.jcreole.marker.JcxSpanMarker;
import com.admc.jcreole.marker.LinkMarker;
import com.admc.jcreole.marker.MarkerMap;
import com.admc.jcreole.marker.MasterDefListMarker;
import com.admc.jcreole.marker.Styler;
import com.admc.jcreole.marker.TocMarker;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.ArrayDeque;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Deque;
import java.util.EnumSet;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.apache.commons.lang.StringEscapeUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang.WordUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

public class CreoleParser
extends Parser {
    static final ParsingTables PARSING_TABLES = new ParsingTables("U9pTLsTt5LKKtNC70OEeW21e2hOGfKHG290W9MA0G4WW91GLK4e01N5Pbn$qaxz3B1G9g1GxDiI23LH00K71YeMcY0fkzxrlmXkVcRoiUIik5lUjFNFEBVkUEJFtdFC0KyyJo4Sxd8FsE1VduNnqmWNeZ2wu45tH3HVXOcGZ1xd8GtTSWajn6Ix75IX57IQY0jDn5ehG0QrGWAlH5$rm3Qv5Vmp2O1IZX2SpPZyxcXqBD3O4FT4HRJ0VYp03ix0GjRWDD#1sZCSOJCCTuXc0cPYCCbJYIbo5NkYDFYY4uVjy9APz8zpEmp4KOp4kOuv9gCcGOoIk2ydb8RaQLPZ0Dq$vwYRS7EahNrvlt5Tnh4lQS6D8dXTwxlcu1RVAZYoqpTY7IMaKHlCTlgKdcetftC8tS7rCzYcOYZf#pcq0tyKqVi2Tt8hKDYmMU$DRLWlpspOmfeVEr5KswjIbzLgFa3m7SqDQQy63qObgfHZMMl6fhJYp$cLB1yMjfkmfqbj9s9EldGhSUUmPdCYMPRTZDiPwEaD355c6wXiVhXDMgfCqKl6Yp9unyPIxw5UXw55feuoYHxMBPRcA8F7iQzYvsBsDqu6Z95AsdiWzrTDVaJFQLpaHlefZNNhR6komEJFJvqzAzeboz2iJVvxx5SPaj$x9jZzIVAFanMpaCQN5OqzbYMxoJelokptwnVJGcRhAVatPQRqs9YIhuWjfjOfyYnzAnjb2AoTqRudGiK5F1xMPpPtxVq7sRsrYh5IuindFDL6OaP2Dhyk3mhQKEGV4LH7CoTN5#cvyQAzAeIeO7wJBHkaJKkovIxWxc1Ey4##UGAyM5eNctYVSAzmlF12omsRG3KdThaEtvlhhT8G#qWDHOshBXLdfEEIZ$D2QrSB3ZN3LX#HrMcDIECQ5nez9BsXa7oyazqtXFw6#tatRYSM96$$cN#x#1u$p5MxWoto9B$85Fi$dk8xhkPPh#0oVvbDSpLMivqe#oHLSpcLSoYVuE1$Zerp2P$adV#Sn$iPVUPH7U9Y7U92$y6V#n1#fRvRVypj#os$uDVUx$VPo3$VnA#xc5$oI7xg#zmARDl8rRdRIsq7F9dxixXy5#kjybRkSz9cmNVXK#8Gx#0E2ffW3$G#jKRohKSIpTSrazOpMlTR1IFQTte3AHb0LWN878zdyPwmuG8c3KGxs1QDixBdAOQo3aMoKwpt1AEyRPNwZ36rKNtYgE2oiR5HpUAe$BAni52SyLICMLZQgLdpLAxxolAzCxojB#yhJllAzfnnjQy44YY8mqC54ZiV5b7y0L85mdm==");
    private static Log log = LogFactory.getLog(CreoleParser.class);
    private static Pattern ImgAsLabelPattern = Pattern.compile("\\s*\\Q{{\\E\\s*(.+?)\\s*\\Q}}\\E\\s*");
    private static Pattern InterWikiPattern = Pattern.compile("([^:]+)?:([^?#]+)(.+)?");
    private static Pattern DoubleQuotedPattern = Pattern.compile("(?s)\"(.+)\"");
    private static Pattern StylerPattern = Pattern.compile("(?s)([-+=])(block|inline|jcxSpan|jcxBlock)\\s+(.*[-\\w])");
    public static final Pattern CssNamesPattern = Pattern.compile("[a-zA-Z_][-\\w]*(?:\\s+[a-zA-Z_][-\\w]*)*");
    public static final Pattern EnumSymbolsPattern = Pattern.compile("[_xaA01iI]{6}");
    public static final Pattern SlashDelimitedPattern = Pattern.compile("([^/]+)?(?:/\\s*(\\S+))?");
    private static final Symbol DUMMY_SYMBOL = new Symbol(null);
    private static final WashedSymbol DUMMY_WASHEDSYMBOL = new WashedSymbol(null);
    public static final String DEFAULT_SECT_ENUM_SYMBOLS = "____xx";
    public static final EnumSet<JCreolePrivilege> DEFAULT_PLUGIN_PRIVILEGES = EnumSet.noneOf(JCreolePrivilege.class);
    private boolean wrote;
    private boolean authorSetSectSymbols;
    private boolean indexAdded;
    private boolean masterDefListAdded;
    private boolean footNotesAdded;
    private InterWikiMapper interWikiMapper;
    private int lastMarkerId;
    private int ewinCount;
    private int tabCount;
    private MarkerMap markers = new MarkerMap();
    private List<String> cssHrefs = new ArrayList<String>();
    private Set<String> hIds = new HashSet<String>();
    private String sectIdPrefix = "jcsect_";
    private String sectEnumSymbols = "____xx";
    private String listEnumSymbols;
    private Character formatResetChar;
    private EnumSet<JCreolePrivilege> jcreolePrivs = DEFAULT_PLUGIN_PRIVILEGES;
    List<Deque<InlineMode>> inlineModeStacks = new ArrayList<Deque<InlineMode>>();
    Deque<InlineMode> inlineModeStack = new ArrayDeque<InlineMode>(3);
    private static final String INDENT_STRING = "  ";
    private final Action[] actions;

    public void setEnumerationFormats(String enumerationFormats) {
        this.setEnumSymbols(enumerationFormats, true);
    }

    public void setEnumSymbols(String symbolString, boolean forSection) {
        if (symbolString == null) {
            throw new NullPointerException("symbolString may not be null");
        }
        if (!EnumSymbolsPattern.matcher(symbolString).matches()) {
            throw new IllegalArgumentException("Malformatted symbolString: " + symbolString);
        }
        if (forSection) {
            this.sectEnumSymbols = symbolString;
        } else {
            this.listEnumSymbols = symbolString;
        }
    }

    public void setPrivileges(EnumSet<JCreolePrivilege> jcreolePrivs) {
        this.jcreolePrivs = jcreolePrivs;
    }

    public EnumSet<JCreolePrivilege> getPrivileges() {
        return this.jcreolePrivs;
    }

    public void setSectIdPrevix(String sectIdPrefix) {
        this.sectIdPrefix = sectIdPrefix;
    }

    public Sections getSectionHeadings() {
        return this.markers.getSectionHeadings();
    }

    private static String autoLabelText(String washedString) {
        return washedString.replaceAll("\\u001a\\w{4}", "").replaceAll("<+[^>]+?>+", "").trim().replaceAll("\\s+", " ");
    }

    private String mkXmlId(String anyString) {
        String safeString = anyString.replaceAll("&\\w+;", "").replaceAll("[^\\w\\s]", " ").replaceAll("\\s+", " ");
        String s = this.sectIdPrefix + WordUtils.capitalizeFully((String)safeString).replaceAll("\\s+", "");
        if (s.length() < 50 && !this.hIds.contains(s)) {
            this.hIds.add(s);
            return s;
        }
        s = this.sectIdPrefix + WordUtils.initials((String)safeString).toLowerCase();
        if (this.hIds.contains(s)) {
            for (int i = 1; i < 10000; ++i) {
                if (this.hIds.contains(s + i)) continue;
                s = s + i;
                break;
            }
        }
        this.hIds.add(s);
        return s;
    }

    public List<String> getCssHrefs() {
        return this.cssHrefs;
    }

    public void setInterWikiMapper(InterWikiMapper interWikiMapper) {
        this.interWikiMapper = interWikiMapper;
    }

    public static StringBuilder joinWashedSymbols(List<? extends WashedSymbol> wtList) {
        StringBuilder sb = new StringBuilder();
        for (WashedSymbol washedSymbol : wtList) {
            sb.append(washedSymbol.toString());
        }
        return sb;
    }

    public static WashedSymbol mkWashedSymbol(List<String> sList) {
        StringBuilder sb = new StringBuilder();
        for (String s : sList) {
            sb.append(s);
        }
        return new WashedSymbol(CreoleParser.clean(sb.toString()));
    }

    private static boolean isAbsoluteUrl(String path) {
        if (path.length() > 0 && path.charAt(0) == '#') {
            return false;
        }
        if (path.length() > 0 && path.charAt(0) == '/') {
            return false;
        }
        try {
            new URL(path);
        }
        catch (MalformedURLException mue) {
            return false;
        }
        return true;
    }

    public static String clean(String s) {
        return StringEscapeUtils.escapeHtml((String)s.replaceAll("&(?=\\w+;)", "\b")).replace('\b', '&');
    }

    private WashedSymbol toggleInlineMode(InlineMode mode, Token tok) {
        if (mode == this.inlineModeStack.peekFirst()) {
            this.inlineModeStack.pop();
            return new WashedSymbol(this.markClose(TagType.INLINE) + "</" + mode.getTagName() + '>');
        }
        if (this.inlineModeStack.contains((Object)mode)) {
            throw new CreoleParseException("Inline modes are tangled", tok);
        }
        this.inlineModeStack.push(mode);
        String cssClass = mode.getCssClass();
        return new WashedSymbol("<" + mode.getTagName() + (cssClass == null ? "" : " class=\"" + cssClass) + this.markInline(mode.getTagName(), cssClass == null, false) + (cssClass == null ? "" : "\"") + '>');
    }

    public WashedSymbol newDl(List<DtSymbol> dtSymbolList) {
        StringBuilder sb = new StringBuilder("<dl").append(this.markBlock("dl", true, false)).append('>');
        for (DtSymbol dtSymbol : dtSymbolList) {
            sb.append("\n  " + dtSymbol.getContent());
        }
        sb.append('\n').append(this.markClose(TagType.BLOCK)).append("</dl>\n");
        return new WashedSymbol(sb.toString());
    }

    public WashedSymbol newListNest(List<LiSymbol> liSymbolList) {
        StringBuilder sb = new StringBuilder();
        int curLevel = 0;
        HashMap<Integer, Character> levelTypeMap = new HashMap<Integer, Character>();
        String tagName = null;
        for (LiSymbol liSymbol : liSymbolList) {
            String string = tagName = liSymbol.getType() == '#' ? "ol" : "ul";
            if (liSymbol.getLevel() == curLevel) {
                if (liSymbol.getType() != ((Character)levelTypeMap.get(curLevel)).charValue()) {
                    throw new CreoleParseException("Illegal type transition", liSymbol);
                }
                sb.append(this.markClose(TagType.BLOCK)).append("</li>\n");
            } else if (liSymbol.getLevel() == curLevel + 1) {
                char globalSymbol;
                char enumSymbol = liSymbol.getEnumSymbol();
                if (enumSymbol == '\u0000' && this.listEnumSymbols != null && (globalSymbol = this.listEnumSymbols.charAt(liSymbol.getLevel() - 1)) != '_' && globalSymbol != 'x') {
                    enumSymbol = globalSymbol;
                }
                sb.append('<').append(tagName).append(this.markBlock(tagName, true, false));
                if (enumSymbol != '\u0000') {
                    sb.append(" type=\"").append(enumSymbol).append('\"');
                }
                sb.append(">\n");
                levelTypeMap.put(liSymbol.getLevel(), Character.valueOf(liSymbol.getType()));
            } else if (liSymbol.getLevel() < curLevel) {
                sb.append(this.markClose(TagType.BLOCK)).append("</li>\n");
                for (int hiLevel = curLevel; hiLevel > liSymbol.getLevel(); --hiLevel) {
                    tagName = ((Character)levelTypeMap.remove(hiLevel)).charValue() == '#' ? "ol" : "ul";
                    sb.append(CreoleParser.indent(hiLevel - 1)).append(this.markClose(TagType.BLOCK)).append("</").append(tagName).append('>').append(this.markClose(TagType.BLOCK)).append("</li>\n");
                }
                if (liSymbol.getType() != ((Character)levelTypeMap.get(liSymbol.getLevel())).charValue()) {
                    throw new CreoleParseException("Illegal type transition", liSymbol);
                }
            } else {
                throw new CreoleParseException("Illegal transition from list level " + curLevel + " to " + liSymbol.getLevel(), liSymbol);
            }
            sb.append(CreoleParser.indent(liSymbol.getLevel())).append("<li").append(this.markBlock("li", true, false)).append('>').append(liSymbol.getContent());
            curLevel = liSymbol.getLevel();
        }
        sb.append(this.markClose(TagType.BLOCK)).append("</li>\n");
        for (int hiLevel = curLevel; hiLevel >= 1; --hiLevel) {
            tagName = ((Character)levelTypeMap.remove(hiLevel)).charValue() == '#' ? "ol" : "ul";
            sb.append(CreoleParser.indent(hiLevel - 1)).append(this.markClose(TagType.BLOCK)).append("</").append(tagName).append('>');
            if (hiLevel > 1) {
                sb.append(this.markClose(TagType.BLOCK)).append("</li>");
            }
            sb.append('\n');
        }
        return new WashedSymbol(sb.toString());
    }

    public WashedSymbol newTabSet(List<TabSymbol> tabSymbolList) {
        StringBuilder sb = new StringBuilder("<div class=\"jcreole_tabset\">\n  <ul>\n");
        for (TabSymbol tabSymbol : tabSymbolList) {
            sb.append("    ").append(tabSymbol.getLabelLi()).append('\n');
        }
        sb.append("  </ul>\n\n");
        for (TabSymbol tabSymbol : tabSymbolList) {
            sb.append(INDENT_STRING).append(tabSymbol.getContentBlock()).append('\n');
        }
        sb.append("</div>\n");
        return new WashedSymbol(sb.toString());
    }

    private String markBlock(String tagName, boolean writeAttr, boolean atomic) {
        BlockMarker bm = new BlockMarker(++this.lastMarkerId, tagName, writeAttr, atomic);
        this.markers.put(this.lastMarkerId, bm);
        return bm.getMarkerString();
    }

    private String markInline(String tagName, boolean writeAttr, boolean atomic) {
        InlineMarker mI = new InlineMarker(++this.lastMarkerId, tagName, writeAttr, atomic);
        this.markers.put(this.lastMarkerId, mI);
        return mI.getMarkerString();
    }

    private String markJcxspan() {
        JcxSpanMarker js = new JcxSpanMarker(++this.lastMarkerId);
        this.markers.put(this.lastMarkerId, js);
        return js.getMarkerString();
    }

    private String markJcxblock() {
        JcxBlockMarker jb = new JcxBlockMarker(++this.lastMarkerId);
        this.markers.put(this.lastMarkerId, jb);
        return jb.getMarkerString();
    }

    private String markClose(TagType targetType) {
        CloseMarker m = new CloseMarker(++this.lastMarkerId, targetType);
        this.markers.put(this.lastMarkerId, m);
        return m.getMarkerString();
    }

    public static String indent(int num) {
        return StringUtils.repeat((String)INDENT_STRING, (int)num);
    }

    private StringBuilder autoCloseInlines() {
        return this.autoCloseInlines(null);
    }

    private StringBuilder autoCloseInlines(InlineMode untilMode) {
        StringBuilder b = new StringBuilder();
        while (this.inlineModeStack.peekFirst() != untilMode) {
            b.append("</").append(this.markClose(TagType.INLINE)).append(this.inlineModeStack.removeFirst().getTagName()).append('>');
        }
        return b;
    }

    public WashedSymbol buildTreeSymbol(List<Symbol> symList) {
        int unfilledCells;
        boolean[][] isHead;
        String[][] cells = new String[symList.size()][];
        int i = -1;
        int cols = -1;
        for (Symbol rSym : symList) {
            cells[++i] = new String[((List)rSym.value).size()];
            if (cells[i].length > cols) {
                cols = cells[i].length;
            }
            int j = -1;
            for (Symbol cTok : (ArrayList)rSym.value) {
                cells[i][++j] = cTok.toString();
            }
        }
        ArrayList<String[]> hCellList = null;
        ArrayList<String[]> bCellList = null;
        ArrayList<boolean[]> bCellIsHead = new ArrayList<boolean[]>();
        for (int r = 0; r < cells.length; ++r) {
            boolean[] ih = new boolean[cells[r].length];
            int headCount = 0;
            for (int c = 0; c < cells[r].length; ++c) {
                if (cells[r][c].length() <= 0 || cells[r][c].charAt(0) != '\u0001') continue;
                ++headCount;
                cells[r][c] = cells[r][c].substring(1);
                ih[c] = true;
            }
            if (headCount == cells[r].length) {
                if (bCellList == null) {
                    if (hCellList == null) {
                        hCellList = new ArrayList<String[]>();
                    }
                    hCellList.add(cells[r]);
                    continue;
                }
                bCellList.add(cells[r]);
                bCellIsHead.add(ih);
                continue;
            }
            if (bCellList == null) {
                bCellList = new ArrayList<String[]>();
            }
            bCellList.add(cells[r]);
            bCellIsHead.add(ih);
        }
        if (bCellList != null && bCellList.size() != bCellIsHead.size()) {
            throw new RuntimeException("Internal Error.  Mismatch between body row count and body-cell-is-head-count.  " + bCellList.size() + " vs. " + bCellIsHead.size());
        }
        String[][] hCells = hCellList == null ? (String[][])null : (String[][])hCellList.toArray((T[])new String[0][]);
        String[][] bCells = bCellList == null ? (String[][])null : (String[][])bCellList.toArray((T[])new String[0][]);
        boolean[][] blArray = isHead = bCellIsHead == null ? (boolean[][])null : (boolean[][])bCellIsHead.toArray((T[])new boolean[0][]);
        if (isHead != null) {
            for (int k = 0; k < isHead.length; ++k) {
                if (isHead[k].length == bCells[k].length) continue;
                throw new RuntimeException("Internal Error.  Mismatch between row cell count and row-is-head-count for row #" + (k + 1) + ".  " + bCells.length + " vs. " + isHead[k].length);
            }
        }
        StringBuilder sb = new StringBuilder("<table class=\"jcreole_data").append(this.markBlock("table", false, false)).append("\">\n");
        if (hCells != null) {
            sb.append(CreoleParser.indent(1)).append("<thead>");
            for (int r = 0; r < hCells.length; ++r) {
                sb.append('\n').append(CreoleParser.indent(2)).append("<tr").append(this.markBlock("tr", true, false)).append("> ");
                for (int c = 0; c < hCells[r].length; ++c) {
                    sb.append("<th");
                    if (c + 1 == hCells[r].length && (unfilledCells = cols - hCells[r].length) > 0) {
                        sb.append(" colspan=\"" + (1 + unfilledCells) + '\"');
                    }
                    sb.append(this.markBlock("th", true, false)).append(this.markClose(TagType.BLOCK)).append('>').append(hCells[r][c]).append("</th>");
                }
                sb.append(this.markClose(TagType.BLOCK)).append(" </tr>");
            }
            sb.append('\n').append(CreoleParser.indent(1)).append("</thead>\n");
        }
        if (bCells != null) {
            sb.append(CreoleParser.indent(1)).append("<tbody>");
            for (int r = 0; r < bCells.length; ++r) {
                sb.append('\n').append(CreoleParser.indent(2)).append("<tr class=\"jcreole_").append(r == 2 * (r / 2) ? "even" : "odd").append('\"').append(this.markBlock("tr", false, false)).append("> ");
                for (int c = 0; c < bCells[r].length; ++c) {
                    sb.append("<t").append(isHead[r][c] ? (char)'h' : 'd');
                    if (c + 1 == bCells[r].length && (unfilledCells = cols - bCells[r].length) > 0) {
                        sb.append(" colspan=\"" + (1 + unfilledCells) + '\"');
                    }
                    String tagName = isHead[r][c] ? "th" : "td";
                    sb.append(this.markBlock(tagName, true, false)).append('>').append(bCells[r][c]).append("</").append(this.markClose(TagType.BLOCK)).append(tagName).append('>');
                }
                sb.append(this.markClose(TagType.BLOCK)).append(" </tr>");
            }
            sb.append('\n').append(CreoleParser.indent(1)).append("</tbody>\n");
        }
        return new WashedSymbol(sb.append(this.markClose(TagType.BLOCK)).append("</table>\n").toString());
    }

    private void require(JCreolePrivilege jcreolePriv, Symbol s) {
        if (!(s instanceof Token)) {
            throw new RuntimeException("Symbol not a Token");
        }
        if (!this.jcreolePrivs.contains((Object)jcreolePriv)) {
            throw new CreoleParseException("You lack the required privilege " + (Object)((Object)jcreolePriv), (Token)s);
        }
    }

    private EntryOrdering orderingFor(String s, Symbol sy) {
        if (s == null) {
            return null;
        }
        try {
            return Enum.valueOf(EntryOrdering.class, s);
        }
        catch (IllegalArgumentException iae) {
            throw new CreoleParseException("Unsupported EntryOrdering: " + s, (Token)sy);
        }
    }

    public CreoleParser() {
        super(PARSING_TABLES);
        this.inlineModeStacks.add(this.inlineModeStack);
        this.actions = new Action[]{new Action(){

            public Symbol reduce(Symbol[] _symbols, int offset) {
                ArrayList<Symbol> lst = new ArrayList<Symbol>();
                lst.add(_symbols[offset + 1]);
                return new Symbol(lst);
            }
        }, new Action(){

            public Symbol reduce(Symbol[] _symbols, int offset) {
                ((ArrayList)_symbols[offset + 1].value).add(_symbols[offset + 2]);
                return _symbols[offset + 1];
            }
        }, new Action(){

            public Symbol reduce(Symbol[] _symbols, int offset) {
                Symbol _symbol_b = _symbols[offset + 1];
                ArrayList _list_b = (ArrayList)_symbol_b.value;
                Symbol[] b = _list_b == null ? new Symbol[]{} : _list_b.toArray(new Symbol[_list_b.size()]);
                StringBuilder sb = CreoleParser.joinWashedSymbols(_list_b);
                log.debug("PRE-PostProcess[" + sb + ']');
                return new WashedSymbol(CreoleParser.this.markers.apply(sb, CreoleParser.this.sectEnumSymbols).toString());
            }
        }, new Action(){

            public Symbol reduce(Symbol[] _symbols, int offset) {
                return new WashedSymbol("");
            }
        }, new Action(){

            public Symbol reduce(Symbol[] _symbols, int offset) {
                return new WashedSymbol("<hr" + CreoleParser.this.markBlock("hr", true, true) + ">\n");
            }
        }, new Action(){

            public Symbol reduce(Symbol[] _symbols, int offset) {
                ArrayList<Symbol> lst = new ArrayList<Symbol>();
                lst.add(_symbols[offset + 1]);
                return new Symbol(lst);
            }
        }, new Action(){

            public Symbol reduce(Symbol[] _symbols, int offset) {
                ((ArrayList)_symbols[offset + 1].value).add(_symbols[offset + 2]);
                return _symbols[offset + 1];
            }
        }, new Action(){

            public Symbol reduce(Symbol[] _symbols, int offset) {
                Symbol h = _symbols[offset + 1];
                Symbol _symbol_f = _symbols[offset + 2];
                ArrayList _list_f = (ArrayList)_symbol_f.value;
                Symbol[] f = _list_f == null ? new Symbol[]{} : _list_f.toArray(new Symbol[_list_f.size()]);
                int hLevel = ((Token)h).getIntParam();
                StringBuilder b = CreoleParser.joinWashedSymbols(_list_f);
                String labelText = CreoleParser.autoLabelText(b.toString());
                String xmlId = CreoleParser.this.mkXmlId(labelText);
                HeadingMarker hm = new HeadingMarker(++CreoleParser.this.lastMarkerId, xmlId, hLevel, labelText);
                if (CreoleParser.this.formatResetChar != null) {
                    hm.setFormatReset(CreoleParser.this.formatResetChar.charValue());
                    CreoleParser.this.formatResetChar = null;
                }
                CreoleParser.this.markers.put(CreoleParser.this.lastMarkerId, hm);
                return new WashedSymbol("<h" + hLevel + " id=\"" + xmlId + '\"' + hm.getMarkerString() + b + CreoleParser.this.autoCloseInlines() + CreoleParser.this.markClose(TagType.BLOCK) + "</span></h" + hLevel + ">\n");
            }
        }, new Action(){

            public Symbol reduce(Symbol[] _symbols, int offset) {
                Symbol text = _symbols[offset + 1];
                return new WashedSymbol("\n");
            }
        }, Action.RETURN, Action.RETURN, Action.RETURN, new Action(){

            public Symbol reduce(Symbol[] _symbols, int offset) {
                Symbol _symbol_l = _symbols[offset + 1];
                WashedSymbol l = (WashedSymbol)((Object)_symbol_l.value);
                return new WashedSymbol(l.toString());
            }
        }, new Action(){

            public Symbol reduce(Symbol[] _symbols, int offset) {
                Symbol t = _symbols[offset + 1];
                return new WashedSymbol(t.toString());
            }
        }, new Action(){

            public Symbol reduce(Symbol[] _symbols, int offset) {
                Symbol _symbol_l = _symbols[offset + 1];
                WashedSymbol l = (WashedSymbol)((Object)_symbol_l.value);
                return new WashedSymbol(l.toString());
            }
        }, new Action(){

            public Symbol reduce(Symbol[] _symbols, int offset) {
                Symbol t = _symbols[offset + 1];
                return new WashedSymbol(t.toString());
            }
        }, new Action(){

            public Symbol reduce(Symbol[] _symbols, int offset) {
                Symbol _symbol_text = _symbols[offset + 1];
                String text = (String)_symbol_text.value;
                int eotIndex = text.indexOf(3);
                if (eotIndex < 0) {
                    return new WashedSymbol("<pre" + CreoleParser.this.markBlock("pre", true, true) + ">\n" + CreoleParser.clean(text.replace("\n }}}", "\n}}}")) + "\n</pre>\n");
                }
                String body = text.substring(0, eotIndex);
                String addlClasses = eotIndex > text.length() - 2 ? "" : " " + text.substring(eotIndex + 1);
                return new WashedSymbol("<pre class=\"prettyprint" + addlClasses + CreoleParser.this.markBlock("pre", false, true) + "\">\n" + CreoleParser.clean(body.replace("\n }}}", "\n}}}")) + "\n</pre>");
            }
        }, new Action(){

            public Symbol reduce(Symbol[] _symbols, int offset) {
                Symbol _symbol_text = _symbols[offset + 1];
                String text = (String)_symbol_text.value;
                if (text.indexOf("--") > -1) {
                    throw new CreoleParseException("'--' prohibited inside of XML comments", (Token)_symbol_text);
                }
                return new WashedSymbol("<!--" + text + "-->");
            }
        }, new Action(){

            public Symbol reduce(Symbol[] _symbols, int offset) {
                Symbol _symbol_text = _symbols[offset + 1];
                String text = (String)_symbol_text.value;
                CreoleParser.this.require(JCreolePrivilege.RAWHTML, _symbol_text);
                return new WashedSymbol(text);
            }
        }, new Action(){

            public Symbol reduce(Symbol[] _symbols, int offset) {
                Symbol _symbol_text = _symbols[offset + 1];
                String text = (String)_symbol_text.value;
                CreoleParser.this.require(JCreolePrivilege.STYLESHEET, _symbol_text);
                CreoleParser.this.cssHrefs.add(CreoleParser.clean(text));
                return DUMMY_WASHEDSYMBOL;
            }
        }, new Action(){

            public Symbol reduce(Symbol[] _symbols, int offset) {
                Symbol _symbol_text = _symbols[offset + 1];
                String text = (String)_symbol_text.value;
                CreoleParser.this.require(JCreolePrivilege.ENUMFORMAT, _symbol_text);
                if (!EnumSymbolsPattern.matcher(text).matches()) {
                    throw new CreoleParseException("Malformatted sectionEnumSymbols: " + text, (Token)_symbol_text);
                }
                if (CreoleParser.this.authorSetSectSymbols) {
                    throw new CreoleParseException("Attempted to set sectionEnumSymbols a second time.  This is a global and location-independent setting, so remove the setting that you don't want.  Use sectionEnumReset directive if you want to change symbols in a location-dependent way.", (Token)_symbol_text);
                }
                CreoleParser.this.authorSetSectSymbols = true;
                CreoleParser.this.sectEnumSymbols = text;
                return DUMMY_WASHEDSYMBOL;
            }
        }, new Action(){

            public Symbol reduce(Symbol[] _symbols, int offset) {
                Symbol _symbol_text = _symbols[offset + 1];
                String text = (String)_symbol_text.value;
                CreoleParser.this.require(JCreolePrivilege.ENUMFORMAT, _symbol_text);
                if (!EnumSymbolsPattern.matcher(text).matches()) {
                    throw new CreoleParseException("Malformatted listEnumSymbols: " + text, (Token)_symbol_text);
                }
                CreoleParser.this.listEnumSymbols = text;
                return DUMMY_WASHEDSYMBOL;
            }
        }, new Action(){

            public Symbol reduce(Symbol[] _symbols, int offset) {
                ArrayList<Symbol> lst = new ArrayList<Symbol>();
                lst.add(_symbols[offset + 1]);
                return new Symbol(lst);
            }
        }, new Action(){

            public Symbol reduce(Symbol[] _symbols, int offset) {
                ((ArrayList)_symbols[offset + 1].value).add(_symbols[offset + 2]);
                return _symbols[offset + 1];
            }
        }, new Action(){

            public Symbol reduce(Symbol[] _symbols, int offset) {
                int lastI;
                Symbol _symbol_f = _symbols[offset + 1];
                ArrayList _list_f = (ArrayList)_symbol_f.value;
                Symbol[] f = _list_f == null ? new Symbol[]{} : _list_f.toArray(new Symbol[_list_f.size()]);
                Symbol text = _symbols[offset + 2];
                int i = -1;
                int firstEntry = -1;
                while (++i < _list_f.size()) {
                    if (!(_list_f.get(i) instanceof EntrySymbol)) continue;
                    if (firstEntry > -1) {
                        throw new CreoleParseException("Multiple Entry Directives in a single paragraph");
                    }
                    firstEntry = i;
                }
                EntrySymbol es = firstEntry > -1 ? (EntrySymbol)((Object)_list_f.remove(firstEntry)) : null;
                StringBuilder b = CreoleParser.joinWashedSymbols(_list_f);
                if (b.charAt(lastI = b.length() - 1) == '\n') {
                    b.deleteCharAt(lastI);
                }
                b.append((CharSequence)CreoleParser.this.autoCloseInlines()).append(CreoleParser.this.markClose(TagType.BLOCK)).append("</p>").insert(0, '>').insert(0, CreoleParser.this.markBlock("p", true, false)).insert(0, "<p");
                if (es == null) {
                    return new WashedSymbol(b.append('\n').toString());
                }
                es.setCleanString(b.toString());
                return es;
            }
        }, new Action(){

            public Symbol reduce(Symbol[] _symbols, int offset) {
                Symbol _symbol_text = _symbols[offset + 1];
                String text = (String)_symbol_text.value;
                CreoleParser.this.require(JCreolePrivilege.TOC, _symbol_text);
                String classNames = null;
                String sectInclusions = null;
                if (text != null) {
                    try {
                        Matcher m = SlashDelimitedPattern.matcher(text);
                        if (!m.matches() || m.groupCount() != 2) {
                            throw new Exception("not [x][/y]");
                        }
                        String str1 = m.group(1);
                        String str2 = m.group(2);
                        if (str1 != null) {
                            classNames = str1.trim();
                        }
                        if (str2 != null) {
                            sectInclusions = str2.trim();
                        }
                        if (classNames != null && !CssNamesPattern.matcher(classNames).matches()) {
                            throw new Exception("bad CSS Name(s)");
                        }
                        if (sectInclusions != null && !EnumSymbolsPattern.matcher(sectInclusions).matches()) {
                            throw new Exception("bad section-inclusion pattern");
                        }
                    }
                    catch (Exception e) {
                        throw new CreoleParseException("TOC spec. not of format '[class names] [/secinclusions]' (" + e.getMessage() + "):  " + text, (Token)_symbol_text);
                    }
                }
                TocMarker tm = new TocMarker(++CreoleParser.this.lastMarkerId, sectInclusions);
                CreoleParser.this.markers.put(CreoleParser.this.lastMarkerId, tm);
                return new WashedSymbol("<div class=\"jcx_toc" + (classNames == null ? "" : " " + classNames) + "\">\n" + tm.getMarkerString() + "\n</div>\n\n");
            }
        }, new Action(){

            public Symbol reduce(Symbol[] _symbols, int offset) {
                Symbol _symbol_text = _symbols[offset + 1];
                String text = (String)_symbol_text.value;
                CreoleParser.this.require(JCreolePrivilege.MASTERDEF, _symbol_text);
                if (CreoleParser.this.masterDefListAdded) {
                    throw new CreoleParseException("Can not have more than one Master def list per Creole document", (Token)_symbol_text);
                }
                CreoleParser.this.masterDefListAdded = true;
                String classNames = null;
                String orderStr = null;
                if (text != null) {
                    try {
                        Matcher m = SlashDelimitedPattern.matcher(text);
                        if (!m.matches() || m.groupCount() != 2) {
                            throw new Exception("not [x][/y]");
                        }
                        String str1 = m.group(1);
                        String str2 = m.group(2);
                        if (str1 != null) {
                            classNames = str1.trim();
                        }
                        if (str2 != null) {
                            orderStr = str2.trim();
                        }
                        if (classNames != null && !CssNamesPattern.matcher(classNames).matches()) {
                            throw new Exception("bad CSS Name(s)");
                        }
                    }
                    catch (Exception e) {
                        throw new CreoleParseException("Index spec. not of format '[class names] [/sortOrder]' (" + e.getMessage() + "):  " + text, (Token)_symbol_text);
                    }
                }
                MasterDefListMarker tm = new MasterDefListMarker(++CreoleParser.this.lastMarkerId, CreoleParser.this.orderingFor(orderStr, _symbol_text));
                CreoleParser.this.markers.put(CreoleParser.this.lastMarkerId, tm);
                return new WashedSymbol("<div class=\"jcx_masterDef" + (classNames == null ? "" : " " + classNames) + "\">\n" + tm.getMarkerString() + "\n</div>\n\n");
            }
        }, new Action(){

            public Symbol reduce(Symbol[] _symbols, int offset) {
                Symbol _symbol_text = _symbols[offset + 1];
                String text = (String)_symbol_text.value;
                CreoleParser.this.require(JCreolePrivilege.FOOTNOTES, _symbol_text);
                if (CreoleParser.this.footNotesAdded) {
                    throw new CreoleParseException("Can not have more than one Foot notes listing per Creole document", (Token)_symbol_text);
                }
                CreoleParser.this.footNotesAdded = true;
                String classNames = null;
                String orderStr = null;
                if (text != null) {
                    try {
                        Matcher m = SlashDelimitedPattern.matcher(text);
                        if (!m.matches() || m.groupCount() != 2) {
                            throw new Exception("not [x][/y]");
                        }
                        String str1 = m.group(1);
                        String str2 = m.group(2);
                        if (str1 != null) {
                            classNames = str1.trim();
                        }
                        if (str2 != null) {
                            orderStr = str2.trim();
                        }
                        if (classNames != null && !CssNamesPattern.matcher(classNames).matches()) {
                            throw new Exception("bad CSS Name(s)");
                        }
                    }
                    catch (Exception e) {
                        throw new CreoleParseException("Index spec. not of format '[class names] [/sortOrder]' (" + e.getMessage() + "):  " + text, (Token)_symbol_text);
                    }
                }
                FootNotesMarker tm = new FootNotesMarker(++CreoleParser.this.lastMarkerId, CreoleParser.this.orderingFor(orderStr, _symbol_text));
                CreoleParser.this.markers.put(CreoleParser.this.lastMarkerId, tm);
                return new WashedSymbol("<div class=\"jcx_footnotes" + (classNames == null ? "" : " " + classNames) + "\">\n" + tm.getMarkerString() + "\n</div>\n\n");
            }
        }, new Action(){

            public Symbol reduce(Symbol[] _symbols, int offset) {
                Symbol _symbol_text = _symbols[offset + 1];
                String text = (String)_symbol_text.value;
                CreoleParser.this.require(JCreolePrivilege.INDEX, _symbol_text);
                if (CreoleParser.this.indexAdded) {
                    throw new CreoleParseException("Can not have more than one index per Creole document", (Token)_symbol_text);
                }
                CreoleParser.this.indexAdded = true;
                String classNames = null;
                String orderStr = null;
                if (text != null) {
                    try {
                        Matcher m = SlashDelimitedPattern.matcher(text);
                        if (!m.matches() || m.groupCount() != 2) {
                            throw new Exception("not [x][/y]");
                        }
                        String str1 = m.group(1);
                        String str2 = m.group(2);
                        if (str1 != null) {
                            classNames = str1.trim();
                        }
                        if (str2 != null) {
                            orderStr = str2.trim();
                        }
                        if (classNames != null && !CssNamesPattern.matcher(classNames).matches()) {
                            throw new Exception("bad CSS Name(s)");
                        }
                    }
                    catch (Exception e) {
                        throw new CreoleParseException("Index spec. not of format '[class names] [/sortOrder]' (" + e.getMessage() + "):  " + text, (Token)_symbol_text);
                    }
                }
                IndexMarker tm = new IndexMarker(++CreoleParser.this.lastMarkerId, CreoleParser.this.orderingFor(orderStr, _symbol_text));
                CreoleParser.this.markers.put(CreoleParser.this.lastMarkerId, tm);
                return new WashedSymbol("<div class=\"jcx_index" + (classNames == null ? "" : " " + classNames) + "\">\n" + tm.getMarkerString() + "\n</div>\n\n");
            }
        }, Action.RETURN, new Action(){

            public Symbol reduce(Symbol[] _symbols, int offset) {
                Symbol s = _symbols[offset + 1];
                if (s instanceof EntrySymbol) {
                    throw new CreoleParseException("Entry Directive not allowed inside a JcxBlock", (Token)s);
                }
                return s;
            }
        }, new Action(){

            public Symbol reduce(Symbol[] _symbols, int offset) {
                ArrayList<Symbol> lst = new ArrayList<Symbol>();
                lst.add(_symbols[offset + 1]);
                return new Symbol(lst);
            }
        }, new Action(){

            public Symbol reduce(Symbol[] _symbols, int offset) {
                ((ArrayList)_symbols[offset + 1].value).add(_symbols[offset + 2]);
                return _symbols[offset + 1];
            }
        }, new Action(){

            public Symbol reduce(Symbol[] _symbols, int offset) {
                int lastI;
                Symbol _symbol_classNames = _symbols[offset + 1];
                String classNames = (String)_symbol_classNames.value;
                Symbol _symbol_f = _symbols[offset + 2];
                ArrayList _list_f = (ArrayList)_symbol_f.value;
                Symbol[] f = _list_f == null ? new Symbol[]{} : _list_f.toArray(new Symbol[_list_f.size()]);
                StringBuilder b = CreoleParser.joinWashedSymbols(_list_f);
                if (b.charAt(lastI = b.length() - 1) == '\n') {
                    b.deleteCharAt(lastI);
                }
                b.append(CreoleParser.this.autoCloseInlines().toString());
                CreoleParser.this.inlineModeStacks.remove(0);
                CreoleParser.this.inlineModeStack = CreoleParser.this.inlineModeStacks.get(0);
                return new WashedSymbol("<div class=\"jcx" + (classNames == null ? "" : " " + classNames) + CreoleParser.this.markJcxblock() + "\">" + b + CreoleParser.this.markClose(TagType.JCXBLOCK) + "</div>\n");
            }
        }, new Action(){

            public Symbol reduce(Symbol[] _symbols, int offset) {
                Symbol _symbol_classNames = _symbols[offset + 1];
                String classNames = (String)_symbol_classNames.value;
                CreoleParser.this.inlineModeStacks.remove(0);
                CreoleParser.this.inlineModeStack = CreoleParser.this.inlineModeStacks.get(0);
                return new WashedSymbol("<div class=\"jcx" + (classNames == null ? "" : " " + classNames) + CreoleParser.this.markJcxblock() + "\">" + CreoleParser.this.markClose(TagType.JCXBLOCK) + "</div>\n");
            }
        }, new Action(){

            public Symbol reduce(Symbol[] _symbols, int offset) {
                Symbol _symbol_text = _symbols[offset + 1];
                String text = (String)_symbol_text.value;
                CreoleParser.this.require(JCreolePrivilege.JCXBLOCK, _symbol_text);
                if (text != null && !CssNamesPattern.matcher(text).matches()) {
                    throw new CreoleParseException("Malformatted class name(s) in JCX directive: " + text, (Token)_symbol_text);
                }
                CreoleParser.this.inlineModeStack = new ArrayDeque<InlineMode>(3);
                CreoleParser.this.inlineModeStacks.add(0, CreoleParser.this.inlineModeStack);
                return _symbol_text;
            }
        }, new Action(){

            public Symbol reduce(Symbol[] _symbols, int offset) {
                Symbol lr = _symbols[offset + 1];
                return CreoleParser.this.buildTreeSymbol(Collections.singletonList(lr));
            }
        }, new Action(){

            public Symbol reduce(Symbol[] _symbols, int offset) {
                ArrayList<Symbol> lst = new ArrayList<Symbol>();
                lst.add(_symbols[offset + 1]);
                return new Symbol(lst);
            }
        }, new Action(){

            public Symbol reduce(Symbol[] _symbols, int offset) {
                ((ArrayList)_symbols[offset + 1].value).add(_symbols[offset + 2]);
                return _symbols[offset + 1];
            }
        }, new Action(){

            public Symbol reduce(Symbol[] _symbols, int offset) {
                Symbol _symbol_rs = _symbols[offset + 1];
                ArrayList _list_rs = (ArrayList)_symbol_rs.value;
                Symbol[] rs = _list_rs == null ? new Symbol[]{} : _list_rs.toArray(new Symbol[_list_rs.size()]);
                Symbol lr = _symbols[offset + 2];
                _list_rs.add(lr);
                return CreoleParser.this.buildTreeSymbol(_list_rs);
            }
        }, new Action(){

            public Symbol reduce(Symbol[] _symbols, int offset) {
                ArrayList<Symbol> lst = new ArrayList<Symbol>();
                lst.add(_symbols[offset + 1]);
                return new Symbol(lst);
            }
        }, new Action(){

            public Symbol reduce(Symbol[] _symbols, int offset) {
                ((ArrayList)_symbols[offset + 1].value).add(_symbols[offset + 2]);
                return _symbols[offset + 1];
            }
        }, Action.RETURN, Action.RETURN, new Action(){

            public Symbol reduce(Symbol[] _symbols, int offset) {
                Symbol c = _symbols[offset + 1];
                Symbol _symbol_l = _symbols[offset + 2];
                ArrayList _list_l = (ArrayList)_symbol_l.value;
                Symbol[] l = _list_l == null ? new Symbol[]{} : _list_l.toArray(new Symbol[_list_l.size()]);
                return new WashedSymbol((((Token)c).getIntParam() == 1 ? "\u0001" : "") + CreoleParser.joinWashedSymbols(_list_l) + CreoleParser.this.autoCloseInlines());
            }
        }, new Action(){

            public Symbol reduce(Symbol[] _symbols, int offset) {
                Symbol ldtitem = _symbols[offset + 1];
                return CreoleParser.this.newDl(Collections.singletonList((DtSymbol)ldtitem));
            }
        }, new Action(){

            public Symbol reduce(Symbol[] _symbols, int offset) {
                ArrayList<Symbol> lst = new ArrayList<Symbol>();
                lst.add(_symbols[offset + 1]);
                return new Symbol(lst);
            }
        }, new Action(){

            public Symbol reduce(Symbol[] _symbols, int offset) {
                ((ArrayList)_symbols[offset + 1].value).add(_symbols[offset + 2]);
                return _symbols[offset + 1];
            }
        }, new Action(){

            public Symbol reduce(Symbol[] _symbols, int offset) {
                Symbol _symbol_ldt = _symbols[offset + 1];
                ArrayList _list_ldt = (ArrayList)_symbol_ldt.value;
                Symbol[] ldt = _list_ldt == null ? new Symbol[]{} : _list_ldt.toArray(new Symbol[_list_ldt.size()]);
                Symbol ldtitem = _symbols[offset + 2];
                ArrayList dtSymbols = _list_ldt;
                dtSymbols.add((DtSymbol)ldtitem);
                return CreoleParser.this.newDl(dtSymbols);
            }
        }, new Action(){

            public Symbol reduce(Symbol[] _symbols, int offset) {
                Symbol llitem = _symbols[offset + 1];
                return CreoleParser.this.newListNest(Collections.singletonList((LiSymbol)llitem));
            }
        }, new Action(){

            public Symbol reduce(Symbol[] _symbols, int offset) {
                ArrayList<Symbol> lst = new ArrayList<Symbol>();
                lst.add(_symbols[offset + 1]);
                return new Symbol(lst);
            }
        }, new Action(){

            public Symbol reduce(Symbol[] _symbols, int offset) {
                ((ArrayList)_symbols[offset + 1].value).add(_symbols[offset + 2]);
                return _symbols[offset + 1];
            }
        }, new Action(){

            public Symbol reduce(Symbol[] _symbols, int offset) {
                Symbol _symbol_litem = _symbols[offset + 1];
                ArrayList _list_litem = (ArrayList)_symbol_litem.value;
                Symbol[] litem = _list_litem == null ? new Symbol[]{} : _list_litem.toArray(new Symbol[_list_litem.size()]);
                Symbol llitem = _symbols[offset + 2];
                ArrayList liSymbols = _list_litem;
                liSymbols.add((LiSymbol)llitem);
                return CreoleParser.this.newListNest(liSymbols);
            }
        }, new Action(){

            public Symbol reduce(Symbol[] _symbols, int offset) {
                Symbol llitem = _symbols[offset + 1];
                return CreoleParser.this.newTabSet(Collections.singletonList((TabSymbol)llitem));
            }
        }, new Action(){

            public Symbol reduce(Symbol[] _symbols, int offset) {
                ArrayList<Symbol> lst = new ArrayList<Symbol>();
                lst.add(_symbols[offset + 1]);
                return new Symbol(lst);
            }
        }, new Action(){

            public Symbol reduce(Symbol[] _symbols, int offset) {
                ((ArrayList)_symbols[offset + 1].value).add(_symbols[offset + 2]);
                return _symbols[offset + 1];
            }
        }, new Action(){

            public Symbol reduce(Symbol[] _symbols, int offset) {
                Symbol _symbol_litem = _symbols[offset + 1];
                ArrayList _list_litem = (ArrayList)_symbol_litem.value;
                Symbol[] litem = _list_litem == null ? new Symbol[]{} : _list_litem.toArray(new Symbol[_list_litem.size()]);
                Symbol llitem = _symbols[offset + 2];
                ArrayList tabSymbols = _list_litem;
                tabSymbols.add((TabSymbol)llitem);
                return CreoleParser.this.newTabSet(tabSymbols);
            }
        }, new Action(){

            public Symbol reduce(Symbol[] _symbols, int offset) {
                Symbol tb = _symbols[offset + 1];
                Symbol _symbol_l = _symbols[offset + 2];
                ArrayList _list_l = (ArrayList)_symbol_l.value;
                Symbol[] l = _list_l == null ? new Symbol[]{} : _list_l.toArray(new Symbol[_list_l.size()]);
                CreoleParser.this.require(JCreolePrivilege.TAB, tb);
                return new TabSymbol((Token)tb, CreoleParser.joinWashedSymbols(_list_l).toString() + CreoleParser.this.autoCloseInlines(), ++CreoleParser.this.tabCount);
            }
        }, new Action(){

            public Symbol reduce(Symbol[] _symbols, int offset) {
                Symbol tb = _symbols[offset + 1];
                Symbol _symbol_l = _symbols[offset + 2];
                ArrayList _list_l = (ArrayList)_symbol_l.value;
                Symbol[] l = _list_l == null ? new Symbol[]{} : _list_l.toArray(new Symbol[_list_l.size()]);
                CreoleParser.this.require(JCreolePrivilege.TAB, tb);
                return new TabSymbol((Token)tb, CreoleParser.joinWashedSymbols(_list_l).toString() + CreoleParser.this.autoCloseInlines(), ++CreoleParser.this.tabCount);
            }
        }, new Action(){

            public Symbol reduce(Symbol[] _symbols, int offset) {
                Symbol li = _symbols[offset + 1];
                Symbol _symbol_l = _symbols[offset + 2];
                ArrayList _list_l = (ArrayList)_symbol_l.value;
                Symbol[] l = _list_l == null ? new Symbol[]{} : _list_l.toArray(new Symbol[_list_l.size()]);
                ((LiSymbol)li).setContent(CreoleParser.joinWashedSymbols(_list_l).toString() + CreoleParser.this.autoCloseInlines());
                return li;
            }
        }, new Action(){

            public Symbol reduce(Symbol[] _symbols, int offset) {
                Symbol li = _symbols[offset + 1];
                Symbol _symbol_l = _symbols[offset + 2];
                ArrayList _list_l = (ArrayList)_symbol_l.value;
                Symbol[] l = _list_l == null ? new Symbol[]{} : _list_l.toArray(new Symbol[_list_l.size()]);
                ((LiSymbol)li).setContent(CreoleParser.joinWashedSymbols(_list_l).toString() + CreoleParser.this.autoCloseInlines());
                return li;
            }
        }, new Action(){

            public Symbol reduce(Symbol[] _symbols, int offset) {
                Symbol dt = _symbols[offset + 1];
                Symbol _symbol_l = _symbols[offset + 2];
                ArrayList _list_l = (ArrayList)_symbol_l.value;
                Symbol[] l = _list_l == null ? new Symbol[]{} : _list_l.toArray(new Symbol[_list_l.size()]);
                ((DtSymbol)dt).setContent(CreoleParser.joinWashedSymbols(_list_l).toString() + CreoleParser.this.autoCloseInlines());
                return dt;
            }
        }, new Action(){

            public Symbol reduce(Symbol[] _symbols, int offset) {
                Symbol dt = _symbols[offset + 1];
                Symbol _symbol_l = _symbols[offset + 2];
                ArrayList _list_l = (ArrayList)_symbol_l.value;
                Symbol[] l = _list_l == null ? new Symbol[]{} : _list_l.toArray(new Symbol[_list_l.size()]);
                ((DtSymbol)dt).setContent(CreoleParser.joinWashedSymbols(_list_l).toString() + CreoleParser.this.autoCloseInlines());
                return dt;
            }
        }, new Action(){

            public Symbol reduce(Symbol[] _symbols, int offset) {
                Symbol listart = _symbols[offset + 1];
                return new LiSymbol((Token)listart);
            }
        }, new Action(){

            public Symbol reduce(Symbol[] _symbols, int offset) {
                Symbol dtstart = _symbols[offset + 1];
                return new DtSymbol((Token)dtstart);
            }
        }, new Action(){

            public Symbol reduce(Symbol[] _symbols, int offset) {
                Symbol s = _symbols[offset + 1];
                if (s instanceof EntrySymbol) {
                    throw new CreoleParseException("Entry Directive not allowed inside a JcxBlock", (Token)s);
                }
                return s;
            }
        }, new Action(){

            public Symbol reduce(Symbol[] _symbols, int offset) {
                Symbol _symbol_formatStr = _symbols[offset + 1];
                String formatStr = (String)_symbol_formatStr.value;
                CreoleParser.this.require(JCreolePrivilege.ENUMFORMAT, _symbol_formatStr);
                if (formatStr.length() != 1) {
                    throw new CreoleParseException("Malformatted format char in sectoinEnumReset directive: " + formatStr, (Token)_symbol_formatStr);
                }
                CreoleParser.this.formatResetChar = Character.valueOf(formatStr.charAt(0));
                return DUMMY_WASHEDSYMBOL;
            }
        }, new Action(){

            public Symbol reduce(Symbol[] _symbols, int offset) {
                Symbol _symbol_text = _symbols[offset + 1];
                String text = (String)_symbol_text.value;
                return new WashedSymbol("<span class=\"jcreole_noWiki" + CreoleParser.this.markInline("span", false, true) + "\">" + CreoleParser.clean(text) + "</span>");
            }
        }, new Action(){

            public Symbol reduce(Symbol[] _symbols, int offset) {
                Symbol _symbol_text = _symbols[offset + 1];
                String text = (String)_symbol_text.value;
                CreoleParser.this.require(JCreolePrivilege.STYLESHEET, _symbol_text);
                CreoleParser.this.cssHrefs.add(CreoleParser.clean(text));
                return DUMMY_WASHEDSYMBOL;
            }
        }, new Action(){

            public Symbol reduce(Symbol[] _symbols, int offset) {
                Symbol _symbol_text = _symbols[offset + 1];
                String text = (String)_symbol_text.value;
                CreoleParser.this.require(JCreolePrivilege.ENUMFORMAT, _symbol_text);
                if (!EnumSymbolsPattern.matcher(text).matches()) {
                    throw new CreoleParseException("Malformatted sectionEnumSymbols: " + text, (Token)_symbol_text);
                }
                if (CreoleParser.this.authorSetSectSymbols) {
                    throw new CreoleParseException("Attempted to set sectionEnumSymbols a second time.  This is a global and location-independent setting, so remove the setting that you don't want.  Use sectionEnumSymbols directive if you want to change symbols in a location-dependent way.", (Token)_symbol_text);
                }
                CreoleParser.this.authorSetSectSymbols = true;
                CreoleParser.this.sectEnumSymbols = text;
                return DUMMY_WASHEDSYMBOL;
            }
        }, new Action(){

            public Symbol reduce(Symbol[] _symbols, int offset) {
                Symbol _symbol_text = _symbols[offset + 1];
                String text = (String)_symbol_text.value;
                CreoleParser.this.require(JCreolePrivilege.ENUMFORMAT, _symbol_text);
                if (!EnumSymbolsPattern.matcher(text).matches()) {
                    throw new CreoleParseException("Malformatted sectionEnumSymbols: " + text, (Token)_symbol_text);
                }
                CreoleParser.this.listEnumSymbols = text;
                return DUMMY_WASHEDSYMBOL;
            }
        }, new Action(){

            public Symbol reduce(Symbol[] _symbols, int offset) {
                Symbol _symbol_text = _symbols[offset + 1];
                String text = (String)_symbol_text.value;
                CreoleParser.this.require(JCreolePrivilege.JCXSPAN, _symbol_text);
                if (text != null && !CssNamesPattern.matcher(text).matches()) {
                    throw new CreoleParseException("Malformatted class name(s) in JCX directive: " + text, (Token)_symbol_text);
                }
                CreoleParser.this.inlineModeStack.push(InlineMode.JCXSPAN);
                return new WashedSymbol("<span class=\"jcx" + (text != null ? " " + text : "") + CreoleParser.this.markJcxspan() + "\">");
            }
        }, new Action(){

            public Symbol reduce(Symbol[] _symbols, int offset) {
                Symbol tok = _symbols[offset + 1];
                StringBuilder caps = CreoleParser.this.autoCloseInlines(InlineMode.JCXSPAN);
                if (InlineMode.JCXSPAN != CreoleParser.this.inlineModeStack.peekFirst()) {
                    throw new CreoleParseException("jcxSpan modes are tangled", (Token)tok);
                }
                CreoleParser.this.inlineModeStack.pop();
                return new WashedSymbol(caps.toString() + CreoleParser.this.markClose(TagType.JCXSPAN) + "</span>");
            }
        }, new Action(){

            public Symbol reduce(Symbol[] _symbols, int offset) {
                ArrayList<Object> lst = new ArrayList<Object>();
                lst.add(_symbols[offset + 1].value);
                return new Symbol(lst);
            }
        }, new Action(){

            public Symbol reduce(Symbol[] _symbols, int offset) {
                ((ArrayList)_symbols[offset + 1].value).add(_symbols[offset + 2].value);
                return _symbols[offset + 1];
            }
        }, new Action(){

            public Symbol reduce(Symbol[] _symbols, int offset) {
                Symbol _symbol_f = _symbols[offset + 1];
                ArrayList _list_f = (ArrayList)_symbol_f.value;
                String[] f = _list_f == null ? new String[]{} : _list_f.toArray(new String[_list_f.size()]);
                return CreoleParser.mkWashedSymbol(_list_f);
            }
        }, new Action(){

            public Symbol reduce(Symbol[] _symbols, int offset) {
                return new WashedSymbol("&nbsp;");
            }
        }, new Action(){

            public Symbol reduce(Symbol[] _symbols, int offset) {
                return new WashedSymbol("<br>");
            }
        }, new Action(){

            public Symbol reduce(Symbol[] _symbols, int offset) {
                Symbol t = _symbols[offset + 1];
                return CreoleParser.this.toggleInlineMode(InlineMode.EMPHASIZED, (Token)t);
            }
        }, new Action(){

            public Symbol reduce(Symbol[] _symbols, int offset) {
                Symbol t = _symbols[offset + 1];
                return CreoleParser.this.toggleInlineMode(InlineMode.STRIKE, (Token)t);
            }
        }, new Action(){

            public Symbol reduce(Symbol[] _symbols, int offset) {
                Symbol t = _symbols[offset + 1];
                return CreoleParser.this.toggleInlineMode(InlineMode.UNDER, (Token)t);
            }
        }, new Action(){

            public Symbol reduce(Symbol[] _symbols, int offset) {
                Symbol t = _symbols[offset + 1];
                return CreoleParser.this.toggleInlineMode(InlineMode.MONO, (Token)t);
            }
        }, new Action(){

            public Symbol reduce(Symbol[] _symbols, int offset) {
                Symbol t = _symbols[offset + 1];
                return CreoleParser.this.toggleInlineMode(InlineMode.SUP, (Token)t);
            }
        }, new Action(){

            public Symbol reduce(Symbol[] _symbols, int offset) {
                Symbol t = _symbols[offset + 1];
                return CreoleParser.this.toggleInlineMode(InlineMode.SUB, (Token)t);
            }
        }, new Action(){

            public Symbol reduce(Symbol[] _symbols, int offset) {
                Symbol t = _symbols[offset + 1];
                return CreoleParser.this.toggleInlineMode(InlineMode.STRONG, (Token)t);
            }
        }, new Action(){

            public Symbol reduce(Symbol[] _symbols, int offset) {
                Matcher matcher;
                Symbol _symbol_text = _symbols[offset + 1];
                String text = (String)_symbol_text.value;
                Token token = (Token)_symbol_text;
                int delimPos = token.getIntParam();
                boolean extWin = false;
                String lText = text;
                if (lText.length() > 0 && lText.charAt(0) == '&') {
                    if (!CreoleParser.this.jcreolePrivs.contains((Object)JCreolePrivilege.WINDOWTARGETING)) {
                        throw new CreoleParseException("You lack the privilege " + (Object)((Object)JCreolePrivilege.WINDOWTARGETING) + " which is required for &-linking");
                    }
                    lText = lText.substring(1);
                    --delimPos;
                    extWin = true;
                }
                String inLbl = delimPos > -1 ? lText.substring(delimPos + 1) : null;
                String urlText = delimPos > -1 ? lText.substring(0, delimPos) : lText;
                String url = null;
                String popupText = null;
                String labelFromMap = null;
                boolean mapLookupFailed = false;
                Matcher matcher2 = matcher = lText.startsWith("http:") || lText.startsWith("https:") || lText.startsWith("ftp:") ? null : InterWikiPattern.matcher(urlText);
                if (matcher != null && matcher.matches()) {
                    if (CreoleParser.this.interWikiMapper == null) {
                        throw new IllegalStateException(String.format("%s  @line:col %d:%d", "Inter- and Intra-wiki links not supported.", token.getLine() + 1, token.getColumn() + 1));
                    }
                    url = CreoleParser.this.interWikiMapper.toPath(matcher.group(1), matcher.group(2));
                    if (url == null) {
                        mapLookupFailed = true;
                        url = CreoleParser.clean(urlText);
                    } else if (matcher.group(3) != null) {
                        url = url + matcher.group(3);
                    }
                    labelFromMap = inLbl == null ? CreoleParser.this.interWikiMapper.toLabel(matcher.group(1), matcher.group(2)) : null;
                } else if (matcher != null) {
                    matcher = DoubleQuotedPattern.matcher(urlText);
                    if (matcher.matches()) {
                        CreoleParser.this.require(JCreolePrivilege.POPUP, _symbol_text);
                        if (inLbl == null) {
                            throw new CreoleParseException("Explicit label is required for popup links");
                        }
                        popupText = CreoleParser.clean(matcher.group(1));
                    } else if (urlText.length() < 1 || urlText.charAt(0) != '#') {
                        DeferredUrlMarker m = new DeferredUrlMarker(++CreoleParser.this.lastMarkerId, CreoleParser.clean(urlText));
                        CreoleParser.this.markers.put(CreoleParser.this.lastMarkerId, m);
                        url = m.getMarkerString();
                    } else {
                        url = CreoleParser.clean(urlText);
                    }
                } else {
                    url = CreoleParser.clean(urlText);
                }
                String label = null;
                if (inLbl == null) {
                    if (!CreoleParser.this.jcreolePrivs.contains((Object)JCreolePrivilege.ABSLINK) && CreoleParser.isAbsoluteUrl(urlText)) {
                        return new WashedSymbol(url);
                    }
                    if (lText.startsWith("#" + CreoleParser.this.sectIdPrefix)) {
                        LinkMarker m = new LinkMarker(++CreoleParser.this.lastMarkerId, CreoleParser.clean(lText));
                        CreoleParser.this.markers.put(CreoleParser.this.lastMarkerId, m);
                        label = m.getMarkerString();
                    } else {
                        label = labelFromMap == null ? CreoleParser.clean(lText) : labelFromMap;
                    }
                } else {
                    if (!CreoleParser.this.jcreolePrivs.contains((Object)JCreolePrivilege.ABSLINK) && CreoleParser.isAbsoluteUrl(urlText)) {
                        throw new CreoleParseException("You lack the privilege " + (Object)((Object)JCreolePrivilege.ABSLINK) + " which is required for Absolute linking");
                    }
                    Matcher m = ImgAsLabelPattern.matcher(inLbl);
                    if (m.matches()) {
                        String imgUrl = null;
                        String altText = null;
                        int pipeIndex = m.group(1).indexOf(124);
                        if (pipeIndex > 0) {
                            imgUrl = m.group(1).substring(0, pipeIndex);
                            altText = m.group(1).substring(pipeIndex + 1);
                        } else {
                            imgUrl = m.group(1);
                        }
                        label = "<img src=\"" + CreoleParser.clean(imgUrl) + '\"' + CreoleParser.this.markInline("img", true, true) + (altText == null ? "" : " alt=\"" + altText + '\"') + '>';
                    } else if (lText.startsWith("#" + CreoleParser.this.sectIdPrefix)) {
                        LinkMarker lm = new LinkMarker(++CreoleParser.this.lastMarkerId, url);
                        CreoleParser.this.markers.put(CreoleParser.this.lastMarkerId, lm);
                        lm.setLabel(CreoleParser.clean(inLbl));
                        label = lm.getMarkerString();
                    } else {
                        label = CreoleParser.clean(inLbl);
                    }
                }
                if (mapLookupFailed) {
                    label = "<span class=\"jcreole_orphanLink\">" + label + "</span>";
                }
                return new WashedSymbol("<a href=\"" + (popupText == null ? url : "javascript:jcreoleInfoPopup(unescape('" + StringEscapeUtils.escapeJavaScript((String)popupText) + "'))") + "\"" + CreoleParser.this.markInline("a", true, true) + (extWin ? " target=\"jc_extwin" + ++CreoleParser.this.ewinCount + '\"' : "") + '>' + label + "</a>");
            }
        }, new Action(){

            public Symbol reduce(Symbol[] _symbols, int offset) {
                Symbol _symbol_text = _symbols[offset + 1];
                String text = (String)_symbol_text.value;
                Token token = (Token)_symbol_text;
                int delimPos = token.getIntParam();
                String url = CreoleParser.clean(delimPos > -1 ? text.substring(0, delimPos) : text);
                String alt = delimPos > -1 ? CreoleParser.clean(text.substring(delimPos + 1)) : null;
                return new WashedSymbol("<img src=\"" + url + (alt == null ? "" : "\" alt=\"" + alt) + '\"' + CreoleParser.this.markInline("img", true, true) + '>');
            }
        }, new Action(){

            public Symbol reduce(Symbol[] _symbols, int offset) {
                Symbol _symbol_text = _symbols[offset + 1];
                String text = (String)_symbol_text.value;
                CreoleParser.this.require(JCreolePrivilege.STYLER, _symbol_text);
                Matcher m = StylerPattern.matcher(text);
                if (!m.matches()) {
                    throw new CreoleParseException("Malformatted Style specification: " + text, (Token)_symbol_text);
                }
                Styler styler = new Styler(++CreoleParser.this.lastMarkerId, m.group(3), m.group(1).charAt(0), m.group(2));
                CreoleParser.this.markers.put(CreoleParser.this.lastMarkerId, styler);
                return new WashedSymbol(styler.getMarkerString());
            }
        }, new Action(){

            public Symbol reduce(Symbol[] _symbols, int offset) {
                Symbol _symbol_text = _symbols[offset + 1];
                String text = (String)_symbol_text.value;
                if (text.indexOf("--") > -1) {
                    throw new CreoleParseException("'--' prohibited inside of XML comments", (Token)_symbol_text);
                }
                return new WashedSymbol("<!--" + text + "-->");
            }
        }, new Action(){

            public Symbol reduce(Symbol[] _symbols, int offset) {
                Symbol _symbol_text = _symbols[offset + 1];
                String text = (String)_symbol_text.value;
                CreoleParser.this.require(JCreolePrivilege.RAWHTML, _symbol_text);
                return new WashedSymbol(text);
            }
        }, new Action(){

            public Symbol reduce(Symbol[] _symbols, int offset) {
                Symbol _symbol_text = _symbols[offset + 1];
                String text = (String)_symbol_text.value;
                int eotIndex = text.indexOf(3);
                if (eotIndex < 0) {
                    return new WashedSymbol("<pre" + CreoleParser.this.markBlock("pre", true, true) + ">\n" + CreoleParser.clean(text.replace("\n }}}", "\n}}}")) + "\n</pre>\n");
                }
                String body = text.substring(0, eotIndex);
                String addlClasses = eotIndex > text.length() - 2 ? "" : " " + text.substring(eotIndex + 1);
                return new WashedSymbol("<pre class=\"prettyprint" + addlClasses + CreoleParser.this.markBlock("pre", false, true) + "\">\n" + CreoleParser.clean(body.replace("\n }}}", "\n}}}")) + "\n</pre>");
            }
        }, new Action(){

            public Symbol reduce(Symbol[] _symbols, int offset) {
                return new WashedSymbol("&#x2014;");
            }
        }, new Action(){

            public Symbol reduce(Symbol[] _symbols, int offset) {
                Symbol _symbol_text = _symbols[offset + 1];
                String text = (String)_symbol_text.value;
                return new EntrySymbol(((Token)_symbol_text).getIntParam() == 0 ? EntryType.FOOTNOTE : EntryType.MASTERDEF, text);
            }
        }, new Action(){

            public Symbol reduce(Symbol[] _symbols, int offset) {
                Symbol _symbol_text = _symbols[offset + 1];
                String text = (String)_symbol_text.value;
                FootNoteRefMarker fnrm = new FootNoteRefMarker(++CreoleParser.this.lastMarkerId, text);
                CreoleParser.this.markers.put(CreoleParser.this.lastMarkerId, fnrm);
                return new WashedSymbol("<a class=\"jcreole_fnref\" href=\"#jcfn" + fnrm.getMarkerString() + "</sup></a>");
            }
        }, new Action(){

            public Symbol reduce(Symbol[] _symbols, int offset) {
                Symbol _symbol_text = _symbols[offset + 1];
                String text = (String)_symbol_text.value;
                IndexedMarker iem = new IndexedMarker(++CreoleParser.this.lastMarkerId, CreoleParser.clean(text));
                CreoleParser.this.markers.put(CreoleParser.this.lastMarkerId, iem);
                return new WashedSymbol("<a class=\"jcreole_indexed\" id=\"jcindexed" + iem.getMarkerString() + "\"></a>");
            }
        }};
    }

    protected Symbol invokeReduceAction(int rule_num, int offset) {
        return this.actions[rule_num].reduce(this._symbols, offset);
    }

    private static enum InlineMode {
        EMPHASIZED("em"),
        STRONG("strong"),
        JCXSPAN("span"),
        MONO("tt"),
        SUP("sup"),
        SUB("sub"),
        STRIKE("del"),
        UNDER("span", "jcreole_underline");

        private String tagName;
        private String cssClass;

        private InlineMode(String tagName, String cssClass) {
            this.tagName = tagName;
            this.cssClass = cssClass;
        }

        private InlineMode(String tagName) {
            this.tagName = tagName;
        }

        public String getTagName() {
            return this.tagName;
        }

        public String getCssClass() {
            return this.cssClass;
        }
    }
}

